### 详细描述

用于读取和旋转硬件定时器的服务。

---
### 类型定义文档

```c
// 定时器状态描述符。

// 此结构体报告有关 Alchemy 时钟的信息，由调用 rt_timer_inquire() 返回。

typedef struct rt_timer_info RT_TIMER_INFO;
```
---
### 函数文档

#### rt_timer_inquire()

> void rt_timer_inquire(RT_TIMER_INFO *info)

查询 Alchemy 时钟。

返回有关 Alchemy 时钟的状态信息。

**参数:**
- `info`: 用于填充时钟信息的结构体地址。

**标签:**
- `unrestricted`

**参考:**
- `rt_timer_info::period`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/timer.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0
#define TASK_NAME "SpinDemo"

RT_TASK demo_task_desc;

void demo_task(void *arg)
{
    RT_TIMER_INFO info;
    rt_timer_inquire(&info);

    printf("Get Timer Information:\n");
    printf("    period:%llu\n",(long long)info.period);
    printf("    date  :%llu\n",(long long)info.date); // info.date ≈ rt_timer_read()
}

int main(int argc, char *argv[])
{
    // 创建任务
    if (rt_task_create(&demo_task_desc, TASK_NAME, TASK_STKSZ, TASK_PRIO, TASK_MODE)) {
        printf("Error creating task\n");
        return 1;
    }

    // 启动任务
    rt_task_start(&demo_task_desc, &demo_task, NULL);

    // 等待任务结束
    rt_task_join(&demo_task_desc);

    // 清理
    rt_task_delete(&demo_task_desc);
    return 0;
}
```

---

#### rt_timer_ns2ticks()

> SRTIME rt_timer_ns2ticks(SRTIME ns)

将纳秒转换为 Alchemy 时钟滴答。

将纳秒数转换为 Alchemy 时钟滴答。此例程处理有符号的纳秒值。这是 `rt_timer_ticks2ns()` 的逆调用。

**参数:**
- `ns`: 要转换的纳秒数。

**返回值:**
- 以 Alchemy 时钟滴答表示的相应值。Alchemy 时钟的分辨率可以在启动应用程序进程时使用 `--alchemy-clock-resolution` 选项设置（默认为 1 纳秒）。

**标签:**
- `unrestricted`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/timer.h>

int main() {
    // 获取当前的时钟滴答数
    RTIME ns = 1000000000;  // 1 秒
    printf("Current ns: %llu\n",ns);

    // 将时钟滴答数转换为纳秒
    RTIME ticks = rt_timer_ns2ticks(ns);
    printf("Current ticks: %llu\n",ticks);

    return 0;
}
```

---

#### rt_timer_read()

> RTIME rt_timer_read(void)

返回当前系统时间。

返回由 xkernel 核心时钟维护的当前时间。

**返回值:**
- 以时钟滴答表示的当前时间（见注释）。

**标签:**
- `unrestricted`

**注释:**
- 时间值是 Alchemy 时钟分辨率的倍数（参见 `--alchemy-clock-resolution` 选项，默认为 1 纳秒）。

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/timer.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0
#define TASK_NAME "SpinDemo"

RT_TASK demo_task_desc;

void demo_task(void *arg)
{
    RTIME start, end, diff;
    RTIME spin_time = 1000000; // 1 毫秒 

    printf("Starting rt_timer_spin() demonstration\n");

    for (int i = 0; i < 5; i++) {
        start = rt_timer_read();
        
        // 使用 rt_timer_spin 实现短暂而精确的延迟
        rt_timer_spin(spin_time);
        
        end = rt_timer_read();
        diff = end - start;

        printf("Iteration %d: Requested spin time: %llu ns, Actual time: %llu ns\n", 
               i + 1, (long long)spin_time, (long long)diff);
    }
}

int main(int argc, char *argv[])
{
    // 创建任务
    if (rt_task_create(&demo_task_desc, TASK_NAME, TASK_STKSZ, TASK_PRIO, TASK_MODE)) {
        printf("Error creating task\n");
        return 1;
    }

    // 启动任务
    rt_task_start(&demo_task_desc, &demo_task, NULL);

    // 等待任务结束
    rt_task_join(&demo_task_desc);

    // 清理
    rt_task_delete(&demo_task_desc);
    return 0;
}
```

---

#### rt_timer_spin()

> void rt_timer_spin(RTIME ns)

忙等待消耗 CPU 周期。

进入一个忙等待循环，持续指定的纳秒数。

由于此服务总是在中断启用的情况下调用，调用者可能会被其他实时活动抢占，因此实际延迟可能会比指定的时间更长。

**参数:**
- `ns`: 以纳秒表示的等待时间。

**标签:**
- `unrestricted`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/task.h>
#include <alchemy/timer.h>

#define TASK_PRIO 50
#define TASK_MODE T_JOINABLE
#define TASK_STKSZ 0
#define TASK_NAME "SpinDemo"

RT_TASK demo_task_desc;

void demo_task(void *arg)
{
    RTIME start, end, diff;
    RTIME spin_time = 1000000; // 1 毫秒 

    printf("Starting rt_timer_spin() demonstration\n");

    for (int i = 0; i < 5; i++) {
        start = rt_timer_read();
        
        // 使用 rt_timer_spin 实现短暂而精确的延迟
        rt_timer_spin(spin_time);
        
        end = rt_timer_read();
        diff = end - start;

        printf("Iteration %d: Requested spin time: %llu ns, Actual time: %llu ns\n", 
               i + 1, (long long)spin_time, (long long)diff);
    }
}

int main(int argc, char *argv[])
{
    // 创建任务
    if (rt_task_create(&demo_task_desc, TASK_NAME, TASK_STKSZ, TASK_PRIO, TASK_MODE)) {
        printf("Error creating task\n");
        return 1;
    }

    // 启动任务
    rt_task_start(&demo_task_desc, &demo_task, NULL);

    // 等待任务结束
    rt_task_join(&demo_task_desc);

    // 清理
    rt_task_delete(&demo_task_desc);
    return 0;
}
```

---

#### rt_timer_ticks2ns()

> SRTIME rt_timer_ticks2ns(SRTIME ticks)

将 Alchemy 时钟滴答转换为纳秒。

此例程将 Alchemy 时钟滴答数转换为纳秒。它处理有符号的纳秒值。这是 `rt_timer_ns2ticks()` 函数的逆调用。

**参数:**
- `ticks`: 要转换的滴答数。

**返回值:**
- 以纳秒表示的相应值。Alchemy 时钟的分辨率可以在启动应用程序进程时使用 `--alchemy-clock-resolution` 选项设置（默认为 1 纳秒）。

**标签:**
- `unrestricted`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <alchemy/timer.h>

int main() {
    // 获取当前的时钟滴答数
    RTIME ticks = rt_timer_read();
    printf("Current ticks: %llu\n",ticks);

    // 将时钟滴答数转换为纳秒
    RTIME ns = rt_timer_ticks2ns(ticks);
    printf("Current ns: %llu\n",ns);

    return 0;
}
```

---